//
//  HistoricalTrendsView.swift
//  death_app Watch App
//
//  Created by Task-Master AI
//

import SwiftUI

struct HistoricalTrendsView: View {
    @StateObject private var historyManager = PredictionHistoryManager.shared
    @State private var selectedTimeframe: HistoricalTimeframe = .week
    @State private var showingExportSheet = false
    
    var body: some View {
        NavigationStack {
            ScrollView {
                VStack(spacing: 16) {
                    timeframePickerView

                    if historyManager.isLoading {
                        loadingView
                    } else if historyManager.historicalData.isEmpty {
                        emptyStateView
                    } else {
                        contentView
                    }
                }
                .padding(.horizontal, 8)
            }
            .navigationTitle("Trends")
            .navigationBarTitleDisplayMode(.inline)
            .toolbar {
                ToolbarItem(placement: .confirmationAction) {
                    Button(action: { showingExportSheet = true }) {
                        Image(systemName: "square.and.arrow.up")
                            .font(.caption)
                    }
                    .disabled(historyManager.historicalData.isEmpty)
                }
            }
            .sheet(isPresented: $showingExportSheet) {
                DataExportView(
                    historicalData: historyManager.historicalData,
                    trendAnalysis: historyManager.trendAnalysis
                )
            }
        }
        .task {
            await historyManager.fetchHistoricalData(timeframe: selectedTimeframe)
        }
        .onChange(of: selectedTimeframe) { newTimeframe in
            Task {
                await historyManager.fetchHistoricalData(timeframe: newTimeframe)
            }
        }
    }
    
    private var timeframePickerView: some View {
        VStack(alignment: .leading, spacing: 8) {
            Text("Time Period")
                .font(.caption)
                .foregroundColor(.secondary)
            
            Picker("Timeframe", selection: $selectedTimeframe) {
                ForEach(HistoricalTimeframe.allCases, id: \.self) { timeframe in
                    Text(timeframe.rawValue)
                        .tag(timeframe)
                }
            }
            .pickerStyle(.automatic)
        }
    }
    
    private var loadingView: some View {
        VStack(spacing: 12) {
            ProgressView()
                .scaleEffect(0.8)
            
            Text("Loading trends...")
                .font(.caption)
                .foregroundColor(.secondary)
        }
        .frame(height: 100)
    }
    
    private var emptyStateView: some View {
        VStack(spacing: 12) {
            Image(systemName: "chart.line.uptrend.xyaxis")
                .font(.title)
                .foregroundColor(.secondary)
            
            Text("No Data Available")
                .font(.headline)
            
            Text("Trends will appear here once you have prediction history for the selected time period.")
                .font(.caption)
                .foregroundColor(.secondary)
                .multilineTextAlignment(.center)
        }
        .padding()
        .frame(height: 150)
    }
    
    private var contentView: some View {
        VStack(spacing: 16) {
            chartSectionView
            
            if let analysis = historyManager.trendAnalysis {
                statisticsSectionView(analysis: analysis)
                contributingFactorsSectionView(analysis: analysis)
            }
        }
    }
    
    private var chartSectionView: some View {
        VStack(alignment: .leading, spacing: 8) {
            SectionHeader(title: "Life Expectancy Chart", icon: "chart.xyaxis.line")
            
            TrendChartView(
                historicalData: historyManager.historicalData,
                timeframe: selectedTimeframe
            )
            .background(
                RoundedRectangle(cornerRadius: 12)
                    .fill(Color.gray)
            )
        }
    }
    
    private func statisticsSectionView(analysis: TrendAnalysis) -> some View {
        VStack(alignment: .leading, spacing: 12) {
            SectionHeader(title: "Key Statistics", icon: "chart.bar")
            
            VStack(spacing: 8) {
                MetricRow.averageTrend(
                    average: analysis.averageLifeExpectancy,
                    timeframe: selectedTimeframe,
                    trend: analysis.trendDirection
                )
                
                Divider()
                
                if let bestDay = analysis.bestDay {
                    MetricRow.bestDay(
                        date: bestDay.timestamp,
                        lifeExpectancy: bestDay.lifeExpectancy,
                        factors: bestDay.contributingFactors ?? []
                    )
                    
                    Divider()
                }
                
                if let worstDay = analysis.worstDay {
                    MetricRow.worstDay(
                        date: worstDay.timestamp,
                        lifeExpectancy: worstDay.lifeExpectancy,
                        factors: worstDay.contributingFactors ?? []
                    )
                    
                    Divider()
                }
                
                MetricRow.totalChange(
                    change: analysis.totalChange,
                    timeframe: selectedTimeframe
                )
            }
            .background(
                RoundedRectangle(cornerRadius: 12)
                    .fill(Color.gray)
                    .padding(.horizontal, -4)
                    .padding(.vertical, -8)
            )
        }
    }
    
    private func contributingFactorsSectionView(analysis: TrendAnalysis) -> some View {
        VStack(alignment: .leading, spacing: 12) {
            SectionHeader(
                title: "Contributing Factors",
                icon: "list.bullet.clipboard",
                subtitle: "Top factors affecting your life expectancy"
            )
            
            if analysis.contributingFactors.isEmpty {
                Text("No significant contributing factors identified")
                    .font(.caption)
                    .foregroundColor(.secondary)
                    .padding()
                    .background(
                        RoundedRectangle(cornerRadius: 8)
                            .fill(Color.gray)
                    )
            } else {
                VStack(spacing: 8) {
                    ForEach(analysis.contributingFactors.indices, id: \.self) { index in
                        let factor = analysis.contributingFactors[index]
                        
                        MetricRow.contributingFactor(factor: factor)
                        
                        if index < analysis.contributingFactors.count - 1 {
                            Divider()
                        }
                    }
                }
                .background(
                    RoundedRectangle(cornerRadius: 12)
                        .fill(Color.gray)
                        .padding(.horizontal, -4)
                        .padding(.vertical, -8)
                )
            }
        }
    }
}

// MARK: - Supporting Views

struct SectionHeader: View {
    let title: String
    let icon: String
    let subtitle: String?
    
    init(title: String, icon: String, subtitle: String? = nil) {
        self.title = title
        self.icon = icon
        self.subtitle = subtitle
    }
    
    var body: some View {
        HStack(spacing: 8) {
            Image(systemName: icon)
                .font(.caption)
                .foregroundColor(.blue)
            
            VStack(alignment: .leading, spacing: 2) {
                Text(title)
                    .font(.caption.bold())
                    .foregroundColor(.primary)
                
                if let subtitle = subtitle {
                    Text(subtitle)
                        .font(.caption2)
                        .foregroundColor(.secondary)
                }
            }
            
            Spacer()
        }
    }
}


#Preview {
    HistoricalTrendsView()
}
